/**
 * Provided with a config file pointing to package resources, bundles an app containing an extension.
 */

// Use it to build a demo app once the Ionic Blank Starter Wontok is built:
// node bin/bundler.js macos/content-filtering-lib/package/config.json

"use strict";

async function doWork() {
  const fs = require("fs");
  const path = require("path");
  const execSync = require("child_process").execSync;

  const cwd = process.cwd();
  const configFilepath = path.resolve(cwd, process.argv[2]);
  console.log("Using config file: ", configFilepath);

  let configRaw = fs.readFileSync(configFilepath);
  let config = JSON.parse(configRaw);
  console.log("Using config:", JSON.stringify(config, null, 2));

  const appName = config["appName"];
  const systemExtensionBundleId = config["systemExtensionBundleId"];
  const systemExtensionExecutableName = config["systemExtensionExecutableName"];

  const macosBundledAppsPath = path.resolve(cwd, "macos-bundled-apps");

  const systemExtensionInfoPlistPath = path.resolve(
    cwd,
    config["systemExtensionInfoPlistPath"]
  );
  const launcherInfoPlistPath = path.resolve(
    cwd,
    config["launcherInfoPlistPath"]
  );
  const systemExtensionEntitlementsPath = path.resolve(
    cwd,
    config["systemExtensionEntitlementsPath"]
  );
  const launcherEntitlementsPath = path.resolve(
    cwd,
    config["launcherEntitlementsPath"]
  );
  const systemExtensionProvisionProfilePath = path.resolve(
    cwd,
    config["systemExtensionProvisionProfilePath"]
  );
  const launcherProvisionProfilePath = path.resolve(
    cwd,
    config["launcherProvisionProfilePath"]
  );
  const launcherUninstallMonitorPath = path.resolve(
    cwd,
    config["launcherUninstallMonitorPath"]
  );
  const launcherIcnsPath = path.resolve(cwd, config["icns"]);

  const targetDir = path.resolve(cwd, config["target"]);
  const sourceDir = path.resolve(cwd, config["appUiPath"]);

  const launcherBundlePath = path.resolve(targetDir, `${appName}.app`);
  const launcherContentsPath = path.resolve(launcherBundlePath, "Contents");
  const launcherPluginPath = path.resolve(
    launcherContentsPath,
    "Helpers",
    "SafeCentral Secure Browser Plugins",
    "aol-plugin.crx"
  );

  const launcherMacosPath = path.resolve(launcherContentsPath, "MacOS");
  const launcherExecutablePath = path.resolve(launcherMacosPath, appName);

  const launcherLibraryPath = path.resolve(launcherContentsPath, "Library");
  const systemeExtensionsPath = path.resolve(
    launcherLibraryPath,
    "SystemExtensions"
  );
  const systemExtensionBundlePath = path.resolve(
    systemeExtensionsPath,
    `${systemExtensionBundleId}.systemextension`
  );
  const systemExtensionContentsPath = path.resolve(
    systemExtensionBundlePath,
    "Contents"
  );
  const systemExtensionMacOSPath = path.resolve(
    systemExtensionContentsPath,
    "MacOS"
  );
  const systemExtensionExecutablePath = path.resolve(
    systemExtensionMacOSPath,
    systemExtensionExecutableName
  );

  const launcherHelpersPath = path.resolve(launcherContentsPath, "Helpers");
  const uiAppBundlePath = path.resolve(launcherHelpersPath, `${appName}.app`);

  const launcherResourcesPath = path.resolve(launcherContentsPath, "Resources");
  const bundledIcnsPath = path.resolve(launcherResourcesPath, "icon.icns");
  const bundledUninstallMonitorPath = path.resolve(
    launcherResourcesPath,
    "uninstall-monitor.plist"
  );
  const finalCleanupScriptPath = path.resolve(
    launcherResourcesPath,
    "final-cleanup"
  );

  console.log("Removing target dir:", targetDir);
  fs.rmSync(targetDir, { recursive: true, force: true });

  console.log("Creating target dir structure");
  fs.mkdirSync(launcherMacosPath, { recursive: true });
  fs.mkdirSync(systemExtensionMacOSPath, { recursive: true });
  fs.mkdirSync(launcherResourcesPath, { recursive: true });
  fs.mkdirSync(launcherHelpersPath, { recursive: true });

  console.log("Copying executables");
  fs.copyFileSync(
    `${__dirname}/../native/executables/app-launcher`,
    launcherExecutablePath
  );
  fs.copyFileSync(
    `${__dirname}/../native/executables/system-extension`,
    systemExtensionExecutablePath
  );
  fs.copyFileSync(
    `${__dirname}/../native/executables/final-cleanup`,
    finalCleanupScriptPath
  );
  console.log("Copying UI App");
  execSync(`ditto "${sourceDir}" "${uiAppBundlePath}"`, { stdio: "inherit" });

  console.log("Copying Other Bundled Apps");
  function getAllDirectoriesSync(directory) {
    return fs
      .readdirSync(directory)
      .filter((file) => fs.statSync(`${directory}/${file}`).isDirectory());
  }
  const directories = getAllDirectoriesSync(macosBundledAppsPath);
  directories.forEach((directory) => {
    const sourcePath = `${macosBundledAppsPath}/${directory}`;
    const destinationPath = `${launcherHelpersPath}/${directory}`;
    try {
      execSync(`ditto "${sourcePath}" "${destinationPath}"`, {
        stdio: "inherit",
      });
      console.log(`Directory ${directory} copied successfully.`);
    } catch (error) {
      console.error(`Error copying directory ${directory}:`, error);
    }
  });

  console.log("Copying Info.plist files");
  fs.copyFileSync(
    systemExtensionInfoPlistPath,
    path.resolve(systemExtensionContentsPath, "Info.plist")
  );
  fs.copyFileSync(
    launcherInfoPlistPath,
    path.resolve(launcherContentsPath, "Info.plist")
  );

  console.log("Copying embedded.provisionprofile files");
  fs.copyFileSync(
    systemExtensionProvisionProfilePath,
    path.resolve(systemExtensionContentsPath, "embedded.provisionprofile")
  );
  fs.copyFileSync(
    launcherProvisionProfilePath,
    path.resolve(launcherContentsPath, "embedded.provisionprofile")
  );

  console.log("Copying icns");
  fs.copyFileSync(launcherIcnsPath, bundledIcnsPath);

  console.log("Copying uninstall monitor");
  fs.copyFileSync(launcherUninstallMonitorPath, bundledUninstallMonitorPath);

  const notarize = require("@electron/notarize").notarize;
  const appleId = process.env["WTQW_MACOS_NOTARIZATION_USERNAME"];
  const appleIdPassword = process.env["WTQW_MACOS_NOTARIZATION_PASSWORD"];
  const ascProvider = process.env["WTQW_MACOS_TEAM_ID"];
  const certName = process.env["WTQW_MACOS_CERTIFICATE_NAME"];

  if (process.env["NOTARIZE"]) {
    console.log("Notarizing UI");
    await notarize({
      appPath: uiAppBundlePath,
      tool: "notarytool",
      appleId,
      appleIdPassword,
      teamId: ascProvider,
    });
  }

  console.log("Signing extension");
  execSync(
    `
      codesign --sign "${certName}" \\
      --force --timestamp --options runtime --verbose \\
      --entitlements "${systemExtensionEntitlementsPath}" \\
      "${path.resolve(
        systemExtensionContentsPath,
        "embedded.provisionprofile"
      )}"
  `,
    { stdio: "inherit" }
  );
  execSync(
    `
      codesign --sign "${certName}" \\
      --force --timestamp --options runtime --verbose \\
      --entitlements "${systemExtensionEntitlementsPath}" \\
      "${systemExtensionBundlePath}"
  `,
    { stdio: "inherit" }
  );
  execSync(
    `
      codesign --verify --deep --strict --verbose=2 "${systemExtensionBundlePath}"
  `,
    { stdio: "inherit" }
  );
  execSync(
    `
      codesign -d --entitlements :- "${systemExtensionBundlePath}"
  `,
    { stdio: "inherit" }
  );
  if (process.env["NOTARIZE"]) {
    console.log("Notarizing extension");
    await notarize({
      appPath: systemExtensionBundlePath,
      tool: "notarytool",
      appleId,
      appleIdPassword,
      teamId: ascProvider,
    });
  }

  console.log("Signing launcher");
  execSync(
    `
      codesign --sign "${certName}" \\
      --force --timestamp --options runtime --verbose \\
      "${launcherPluginPath}"
  `,
    { stdio: "inherit" }
  );
  execSync(
    `
      codesign --sign "${certName}" \\
      --force --timestamp --options runtime --verbose \\
      --entitlements "${launcherEntitlementsPath}" \\
      "${path.resolve(launcherContentsPath, "embedded.provisionprofile")}"
  `,
    { stdio: "inherit" }
  );
  execSync(
    `
      codesign --sign "${certName}" \\
      --force --timestamp --options runtime --verbose \\
      --entitlements "${launcherEntitlementsPath}" \\
      "${launcherBundlePath}"
  `,
    { stdio: "inherit" }
  );
  execSync(
    `
      codesign --verify --deep --strict --verbose=2 "${launcherBundlePath}"
  `,
    { stdio: "inherit" }
  );
  execSync(
    `
      codesign -d --entitlements :- "${launcherBundlePath}"
  `,
    { stdio: "inherit" }
  );
  if (process.env["NOTARIZE"]) {
    console.log("Notarizing Launcher");
    await notarize({
      appPath: launcherBundlePath,
      tool: "notarytool",
      appleId,
      appleIdPassword,
      teamId: ascProvider,
    });
  }

  // Tree is no longer available by default.
  // execSync(
  //   `
  //     tree -L 5 ${targetDir}
  // `,
  //   { stdio: "inherit" }
  // );
}

doWork().then(() => console.log("Bundler is done!"));
